<?php
namespace App\Http\Controllers\Apis;

use App\Http\Controllers\Controller;
use App\Models\Comment;
use App\Models\Post;
use App\Models\ReportComment;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class UserCommentController extends Controller
{
    const COMMENT_VALIDATION = 'required|string|max:500';

    public function store(Request $request)
    {
        $request->validate([
            'comment'   => self::COMMENT_VALIDATION,
            'post_id'   => 'required|exists:posts,id',
            'replay_id' => 'nullable|exists:comments,id',
        ]);

        $user = Auth::user();

        if ($user) {
            $comment = Comment::create([
                'user_id'   => $user->id,
                'post_id'   => $request->post_id,
                'parent_id' => $request->replay_id,
                'comment'   => $request->comment,
            ]);

            $post = Post::find($request->post_id);
            $post->increment('comment');

            $comment->load('user');

            $data = [
                'post_id'   => $request->post_id,
                'parent_id' => $request->replay_id ?? '0',
                'comment'   => $request->comment,
                'status'    => '1',
            ];

            return response()->json([
                'error'   => false,
                'message' => "Comment store successfully.",
                'data'    => $data,
            ], 200);
        } else {

            $data = [
                'post_id'   => $request->post_id,
                'parent_id' => $request->replay_id ?? '0',
                'comment'   => $request->comment,
                'status'    => '0',
            ];

            return response()->json([
                'error'   => false,
                'message' => "Unauthorized user..",
                'data'    => $data,
            ], 201);
        }
    }

    public function show($id)
    {
        $postId        = $id;
        $perPage       = request()->get('per_page', 3);
        $currentUserId = Auth::user()->id;
        $currentUser   = User::find($currentUserId);
        $token         = $currentUser->createToken('AuthToken')->plainTextToken;
        try {

            $parentComments = Comment::with('user')
                ->where('post_id', $postId)
                ->whereNull('parent_id')
                ->orderBy('id', 'desc')
                ->paginate($perPage);

            $parentCommentIds = $parentComments->pluck('id')->toArray();

            $replies = Comment::with('user')
                ->where('post_id', $postId)
                ->whereIn('parent_id', $parentCommentIds)
                ->orderBy('id', 'desc')
                ->get();

            // ✅ Get removed comment IDs
            $removedCommentIds = ReportComment::where('status', 'removed')
                ->whereIn('comment_id', array_merge($parentCommentIds, $replies->pluck('id')->toArray()))
                ->pluck('comment_id')
                ->toArray();

            $commentsById      = [];
            $organizedComments = [];
            // Why?
            $defaultProfileImage = url('public/front_end/classic/images/default/profile-avatar.jpg');

            foreach ($parentComments as $comment) {
                $isRemoved        = in_array($comment->id, $removedCommentIds);
                $userProfileImage = $comment->user->profile ?? $defaultProfileImage;

                // ✅ Handle removed comment message
                $commentText = $comment->comment;
                if ($isRemoved) {
                    $commentText = ($comment->user_id == $currentUserId)
                        ? 'Your comment has been removed by the admin.'
                        : 'This comment has been removed by the admin.';
                }
                $commentData = [
                    'id'         => $comment->id,
                    'text'       => $commentText,
                    'created_at' => Carbon::parse($comment->created_at)->diffForHumans(),
                    'is_removed' => $isRemoved,
                    'user'       => [
                        'id'      => $comment->user->id,
                        'name'    => $comment->user->name,
                        'profile' => $userProfileImage,
                        'token'   => $token,
                    ],
                    'replies'    => 0,
                ];
                $commentsById[$comment->id] = $commentData;
            }

            // Attach replies count to each parent comment
            foreach ($replies as $reply) {
                if (isset($commentsById[$reply->parent_id])) {
                    $commentsById[$reply->parent_id]['replies']++;
                }
            }

            // Reorganize comments by parent structure
            $organizedComments = array_values($commentsById);
            $count             = Comment::where('post_id', $postId)->count();
            // Prepare response
            $response = [
                "error"   => false,
                "message" => "Comments fetched successfully",
                "data"    => [
                    'count'   => $count,
                    'comment' => $organizedComments,
                ],
            ];

            return response()->json($response);
        } catch (\Throwable $th) {
            return response()->json(['error' => true, "message" => $th->getMessage(), "data" => ["detailed_error" => $th]]);
        }

    }

    // Take this for reference
    public function replayShow($postId, $parentId)
    {
        $page          = request()->get('page', 1);
        $perPage       = request()->get('per_page', 10);
        $offset        = ($page - 1) * $perPage;
        $currentUserId = Auth::user()->id;
        $currentUser   = User::find($currentUserId);
        $token         = $currentUser->createToken('AuthToken')->plainTextToken;
        try {
            $comments = Comment::with('user')
                ->where('post_id', $postId)
                ->where('parent_id', $parentId)
                ->orderBy('id', 'desc')
                ->offset($offset)
                ->limit($perPage)
                ->get();

            $removedCommentIds = ReportComment::where('status', 'removed')
                ->whereIn('comment_id', $comments->pluck('id')->toArray())
                ->pluck('comment_id')
                ->toArray();

            $commentsById        = [];
            $defaultProfileImage = url('public/front_end/classic/images/default/profile-avatar.jpg');

            foreach ($comments as $comment) {
                $isRemoved        = in_array($comment->id, $removedCommentIds);
                $userProfileImage = $comment->user->profile ?? $defaultProfileImage;

                $commentText = $comment->comment;
                if ($isRemoved) {
                    $commentText = ($comment->user_id == $currentUserId)
                        ? 'Your comment has been removed by the admin.'
                        : 'This comment has been removed by the admin.';
                }
                $commentData = [
                    'id'         => $comment->id,
                    'user_id'    => $comment->user_id,
                    'post_id'    => $comment->post_id,
                    'parent_id'  => $comment->parent_id,
                    'replies'    => $comment->replies,
                    'comment'    => $commentText,
                    'is_removed' => $isRemoved,
                    'created_at' => Carbon::parse($comment->created_at)->diffForHumans(),
                    'user'       => [
                        'id'      => $comment->user->id,
                        'name'    => $comment->user->name,
                        'profile' => $userProfileImage,
                        'token'   => $token,
                    ],
                ];

                $commentsById[$comment->id] = $commentData;
            }

            $organizedComments = array_values($commentsById);

            $response = [
                "error"   => false,
                "message" => "Comments fetched successfully",
                "data"    => $organizedComments,
            ];
            return response()->json($response);
        } catch (\Throwable $th) {
            return response()->json(['error' => true, "message" => $th->getMessage(), "data" => ['detailed_message' => $th]], 500);
        }

    }

    public function update(Request $request)
    {
        $request->validate([
            'comment' => self::COMMENT_VALIDATION,
            'id'      => 'required|integer', // Just validate it's required and integer
        ]);

        if (Auth::check()) {
            try {
                $comment          = Comment::findOrFail($request->id); // This will throw 404 if not found
                $comment->comment = $request->comment;
                $comment->save(); // save() is more appropriate than update() here

                return response()->json([
                    'error'      => false,
                    'message'    => "comment updated successfully",
                    'comment_id' => $request->id,
                ]);
            } catch (ModelNotFoundException $e) {
                return response()->json([
                    'error'   => true,
                    'message' => "Comment not found",
                ], 404);
            }
        } else {
            return response()->json([
                'error'   => true,
                'message' => "Unauthorized user..",
            ], 401);
        }
    }

    public function destroy($id)
    {
        if (Auth::check()) {

            if ($id) {
                $this->deleteChildComment($id);
            }
            $comments = Comment::find($id);
            if ($comments) {
                $post = Post::find($comments->post_id);
                $comments->delete();
                if ($post->comments > 0) {
                    $post->decrement('comment');
                }
                $message = 'Comment Deleted Successfully.';

            } else {
                $message = 'Comment not found.';
                print_r($message);exit;
            }

        } else {
            $message = 'Unauthorized user.';
        }
        return response()->json([
            'error'      => false,
            'message'    => $message,
            'comment_id' => $id ?? "",
        ]);
    }

    public function reportComment(Request $request)
    {
        try {
            $user = Auth::user();

            if (! $user) {
                return response()->json([
                    'success' => false,
                    'message' => 'You must be logged in to report a comment.',
                ], 401);
            }

            // Server-side validation
            $request->validate([
                'comment_id'     => 'required|exists:comments,id',
                'report_type_id' => 'nullable|required_without:other_type|exists:report_types,id',
                'report'         => 'required|string|max:500',
                'is_other'       => 'nullable|boolean',
                'other_type'     => 'nullable|required_if:is_other,1|string|max:255',
            ]);

            // Prevent duplicate report by same user on same comment
            $existing = ReportComment::where('user_id', $user->id)
                ->where('comment_id', $request->comment_id)
                ->first();

            if ($existing) {
                return response()->json([
                    'success' => false,
                    'message' => 'You have already reported this comment.',
                ], 409);
            }

            // Store report
            $report = ReportComment::create([
                'user_id'        => $user->id,
                'comment_id'     => $request->comment_id,
                'report_type_id' => $request->is_other ? null : $request->report_type_id,
                'report'         => $request->report,
                'is_other'       => $request->is_other ?? false,
                'other_type'     => $request->is_other ? $request->other_type : null,
                'status'         => 'pending',
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Thank you! Your report has been submitted successfully.',
                'data'    => $report,
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            // Return validation errors for JS (iziToast)
            return response()->json([
                'success' => false,
                'errors'  => $e->errors(),
            ], 422);
        } catch (\Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function deleteChildComment($id)
    {
        $comments = Comment::where('parent_id', $id)->orderBy('id', 'desc')->get();
        if ($comments) {
            foreach ($comments as $comment) {
                $post = Post::find($comment->post_id);

                $comment->delete();
                if ($post->comment > 0) {
                    $post->decrement('comment');
                }
            }
            $message = "parent comment deleted successfully";
        } else {
            $message = "parent comment not found";
        }
        return $message;
    }
}
